// Copyright (c) 2020 Robotics and Artificial Intelligence Lab, KAIST
//
// Any unauthorized copying, alteration, distribution, transmission,
// performance, display or use of this material is prohibited.
//
// All rights reserved.

#include "raisin_example_plugin/example.hpp"

namespace raisin
{
namespace plugin
{
ExamplePlugin::ExamplePlugin(
  raisim::World & world, raisim::RaisimServer & server,
  raisim::World & worldSim, raisim::RaisimServer & serverSim, GlobalResource & globalResource)
: Plugin(world, server, worldSim, serverSim, globalResource)
{
  pluginType_ = PluginType::CUSTOM; // select plugin type

  worldHub_.removeObject(worldHub_.getObject("terrain"));
  std::vector<double> heightVec;
  heightVec.resize(4);
  auto heightMap = worldHub_.addHeightMap(
    2, 2, 4.0, 4.0,
    0, 0, heightVec);
  heightMap->setName("terrain");

  gc_.setZero(19);
  gv_.setZero(18);
  linAccB_.setZero(3);
  angVelB_.setZero(3);
  quat_.setZero(4);


  logIdx_ = dataLogger_.initializeAnotherDataGroup(
    "raisin_example_plugin",
    "gc_", gc_,
    "gv_", gv_,
    "linAccB_", linAccB_,
    "angVelB_", angVelB_,
    "quat_", quat_,
    "loopTime_", loopTime_);
}

ExamplePlugin::~ExamplePlugin()
{
  worldHub_.removeObject(raisimHeightMap_);
  auto ground = worldHub_.addGround();
  ground->setName("terrain");
}

bool ExamplePlugin::advance()
{
  auto sectionTimer = SectionTimer(); // feature to measure elapsed time

  // 1. you can get sensor measurements and estimated states
  auto depthCam = robotHub_->getSensorSet("d430_front")->getSensor<raisim::DepthCamera>("depth");
  depthCam->lockMutex();
  auto depthArray = depthCam->getDepthArray();
  depthCam->unlockMutex();

  auto imu = robotHub_->getSensorSet("base_imu")->getSensor<raisim::InertialMeasurementUnit>("imu");
  imu->lockMutex();
  linAccB_ = imu->getLinearAcceleration();
  angVelB_ = imu->getAngularVelocity();
  quat_ = imu->getOrientation().e();
  imu->unlockMutex();

  robotHub_->lockMutex();
  robotHub_->getState(gc_, gv_);
  robotHub_->unlockMutex();


  // 2. or you can estimate the environment(for example, height mapping)
  auto heightMap = dynamic_cast<raisim::HeightMap *>(worldHub_.getObject("terrain"));
  std::vector<double> heightVec;
  /// TODO: implement your own height mapping
  heightVec.resize(4);
  heightMap->update(
    0, 0,
    4.0, 4.0, heightVec);

  // 3. or you can estimate robot's state
  Eigen::VectorXd gc;
  Eigen::VectorXd gv;

  /// TODO: implement your own state estimator
  robotSim_->lockMutex();
  robotSim_->getState(gc, gv); // this should not be used on a real robot. Just for simplicity.
  robotSim_->unlockMutex();

  robotHub_->lockMutex();
  robotHub_->setState(gc, gv); // you can estimate robot's state
  robotHub_->unlockMutex();

  sectionTimer.measure(loopTime_); // measure the time of the advance loop

  dataLogger_.append(
    logIdx_, gc_, gv_, linAccB_, angVelB_, quat_, loopTime_);

  return true;
}
extern "C" Plugin * create(
  raisim::World & worldHub, raisim::RaisimServer & serverHub,
  raisim::World & worldSim, raisim::RaisimServer & serverSim, GlobalResource & globalResource)
{
  return new ExamplePlugin(worldHub, serverHub, worldSim, serverSim, globalResource);
}

extern "C" void destroy(Plugin * p)
{
  delete p;
}

}     // plugin
} // raisin
